function values_and_averages3(data,thelabel,widthFactor,style)
% VALUES_AND_AVERAGES3(DATA)
% VALUES_AND_AVERAGES3(DATA,THELABEL,widthFactor)
% VALUES_AND_AVERAGES3(DATA,THELABEL,widthFactor,STYLE)
%
% Visualizes all points in the data, columnwise, slightly differently from VALUES_AND_AVERAGES2.
% Hopefully this version will be better.
% WIDTHFACTOR regulates how much the clouds are compacted: the higher the value, the thinner
%   the cloud. Default value of 10.
% STYLE may be either 1 (or omitted): default, for me, or 2 (for publications)

% Apr 22 2013: Created.
% Apr 20 2013: Update to a workable form.
% May 24 2013: New style created.
% Jun 17 2013: Some minor change of the "Publication" (style=2) format.
% Nov 26 2013: widthFactor can not be omitted ([]).
% Jan 24 2014: new styles: averages & SEM.
% Feb 13 2014: Default style changed from 20-80 to 25-75%.
% Jan 06 2015: Help

% This first function is actually just a wrapper for the real function below.
if(nargin>0) % Normal
    if(nargin<2)
        thelabel = '';
    end
    if(nargin<3)
        widthFactor = 10;
    end
    if(isempty(widthFactor))
        widthFactor = 10;
    end
    if(nargin<4)
        style = 'pub';
    end
    switch style
        case 'full'
            do(data,widthFactor);
        case 'pub'
            do2(data,widthFactor);      % Publication style
        case 'mean'
            fprintf('Warning: using sd instead of IQR, and it works incorrectly if outliers are cut out!\n');
            do2(data,widthFactor,1);    % Publication style with means
        case 'av'
            do_av(data,0);              % Averages and STD instead of means and percentiles
        case 'sem'
            do_av(data,1);              % SEM
    end
    set(gca,'XTick',1:length(thelabel),'XTickLabel',thelabel);
    
else % Test
    n = 30;
    m = 5;
    data = 1./(rand(n,m)+0.1);
    data = data+randn(n,m);
    data([2 11 end-7]) = NaN;
    data = bsxfun(@plus,data,rand(1,m)*3);
    
    figure;
    set(gcf,'Position',get(gcf,'Position').*[0.1 0.6 2.2 0.8]);
    subplot(1,3,1); values_and_averages2(data); title('Old');
    subplot(1,3,2); do(data,10); title('New 1');    
    subplot(1,3,3); do2(data,10); title('New 2');    
end

end


function do(data,widthFactor)
% Key function. This one is for "daily use".

i = ~isnan(data(:));
m1 = min(data(i));
m2 = max(data(i));
[n,m] = size(data);

[t1,t2,tick] = tick_value(m1,m2);
nBins = 50;
binData = round((data-m1)/(m2-m1)*(nBins-1))+1;
data2 = binData/nBins*(m2-m1)+m1;

ha = gca;
hold on;

for(iC=1:m)    
    for(iBin=1:nBins)
        loc = data(binData(:,iC)==iBin,iC);
        plot(iC*ones(size(loc))+((1:length(loc))'-(length(loc)+1)/2)/widthFactor,loc,'o',...
            'MarkerFaceColor',[1 1 1]*0.9,'MarkerEdgeColor','none','MarkerSize',5); %'LineSmoothing','on');
    end
    g = ~isnan(data(:,iC));
    plot(iC*[1 1],quantile(data(g,iC),[0.1 0.9]),'r-');
    plot(iC,quantile(data(g,iC),0.75),'^','MarkerFaceColor','r','MarkerEdgeColor','none');
    plot(iC,quantile(data(g,iC),0.25),'v','MarkerFaceColor','r','MarkerEdgeColor','none');
    plot(iC,median(data(g,iC)),'s','MarkerFaceColor','k','MarkerEdgeColor','none');
end

% boxplot(data,'symbol','','notch','on');

hold off;
xlim([0 m+1]);

end


function do2(data,widthFactor,meanFlag)
% Key function ALTERNATIVE VERSION. Publication style

if(nargin<3)
    meanFlag = 0;
end

xspread = 0.12*[-1 1];
level1 = 0.25;      % Normal distribution leaves 16% left of the 1std mark, but it would be harder to explain
level2 = level1;

i = ~isnan(data(:));
m1 = min(data(i));
m2 = max(data(i));
[n,m] = size(data);

[t1,t2,tick] = tick_value(m1,m2);
nBins = 50;
binData = round((data-m1)/(m2-m1)*(nBins-1))+1;
data2 = binData/nBins*(m2-m1)+m1;

ha = gca;
hold on;

for(iC=1:m)    
    for(iBin=1:nBins)
        loc = data(binData(:,iC)==iBin,iC);
        plot(iC*ones(size(loc))+((1:length(loc))'-(length(loc)+1)/2)/widthFactor,loc,'o',...
            'MarkerFaceColor',[1 1 1]*0.9,'MarkerEdgeColor','none','MarkerSize',5); %'LineSmoothing','on');
    end
    g = ~isnan(data(:,iC));
    if(~meanFlag)
        plot(iC*[1 1],quantile(data(g,iC),[level1 (1-level1)]),'k-');
        plot(iC+xspread,quantile(data(g,iC),level2)*[1 1],'k-');
        plot(iC+xspread,quantile(data(g,iC),1-level2)*[1 1],'k-');
        plot(iC,median(data(g,iC)),'s','MarkerFaceColor','g','MarkerEdgeColor','k','MarkerSize',8);
    else        
        dm = mean(data(g,iC));
        el = dm-std(data(g,iC));
        eu = dm+std(data(g,iC));
        (eu-el)/2
        plot(iC*[1 1],[el eu],'k-');
        plot(iC+xspread,el*[1 1],'k-');
        plot(iC+xspread,eu*[1 1],'k-');
        plot(iC,dm,'s','MarkerFaceColor','g','MarkerEdgeColor','k','MarkerSize',8);
    end
end

% boxplot(data,'symbol','','notch','on');

hold off;
xlim([0 m+1]);
set(gcf,'Color',[1 1 1]);

end


function [T1, T2, Tick] = tick_value(M1, M2, N);
% [T1, T2, Tick] = Tick_Value(M1, M2);
% [T1, T2, Tick] = Tick_Value(M1, M2, N);
% Finds the nearest Tick Values (for plotting nice graphs)
%       M1 - min
%       M2 - max (or contrawise - it doesn't matter)
%       N - desirable number of tics (result number of ticks is >= N)

if(M1==M2)
    if(M1==0)
        M2 = 1;
    else
        M1 = M1*.9;
        M2 = M2*1.1;
    end
end
if M1>M2
    [M1 M2]=deal(M2,M1);
end
if(nargin<3)
    N = 5;
end

Tick = 10^floor(log10((M2-M1)/N));
% The lower estimation of a tick. 'Tick resolution' we can say.

N1 = floor(M1/Tick);
N2 = ceil(M2/Tick);

if((N2-N1)>=5*(N-1))
    Tick = Tick*5;
elseif((N2-N1)>=2*(N-1))
    Tick = Tick*2;
end
% This isn't empirical, that's rather good.

N1 = floor(M1/Tick);
N2 = ceil(M2/Tick);

T1 = N1*Tick;
T2 = N2*Tick;

end



function do_av(data,flagSem)
% Key function. This one is for "daily use".

ha = gca;
hold on;

[n,m] = size(data);
xspread = 0.12*[-1 1];

for(iC=1:m)    
    g = ~isnan(data(:,iC));
    m = mean(data(g,iC));    
    s = std(data(g,iC));
    if(flagSem)
        s = s/sqrt(length(data(g,iC)));
    end
    plot(iC,m,'s','MarkerFaceColor','g','MarkerEdgeColor','k','MarkerSize',10);
    plot(iC*[1 1],m+[-s s],'k-');
    plot(iC+xspread,(m+s)*[1 1],'k-');
    plot(iC+xspread,(m-s)*[1 1],'k-');    
end

% boxplot(data,'symbol','','notch','on');

hold off;
% xlim([0 m+1]);

end